import Chat from '../models/Chat.js';
import User from '../models/User.js';
import Groq from 'groq-sdk';
import { 
  loadReferenceData, 
  searchReferenceData, 
  formatReferenceForPrompt,
  formatSourcesForDisplay,
  getAvailableLaws,
  autoDetectRelevantLaws
} from '../services/referenceDataService.js';
import {
  searchExternalSources,
  formatExternalSourcesForPrompt,
  formatExternalSourcesForDisplay,
  shouldUseExternalSearch
} from '../services/externalSearchService.js';
import {
  getSystemPromptForMode,
  getChatModeDisplayName,
  getChatModeDescription
} from '../services/chatModeService.js';

// Cache reference data (reload on server restart)
let cachedReferenceData = null;
const loadCachedReferenceData = () => {
  if (!cachedReferenceData) {
    cachedReferenceData = loadReferenceData();
  }
  return cachedReferenceData;
};

// Initialize Groq client lazily to ensure env vars are loaded
const getGroqClient = () => {
  if (!process.env.GROQ_API_KEY) {
    throw new Error('GROQ_API_KEY is not set in environment variables');
  }
  return new Groq({
    apiKey: process.env.GROQ_API_KEY,
  });
};

// Check if question is legal-related
const isQuestionLegal = (question) => {
  const questionLower = question.toLowerCase();
  const legalKeywords = [
    'قانون', 'مادة', 'ماد', 'حق', 'حقوق', 'التزام', 'عقد', 'عقود',
    'جريمة', 'عقوبة', 'عقاب', 'دعوى', 'قضية', 'حكم', 'قاضي', 'محكمة',
    'زواج', 'طلاق', 'ميراث', 'وصية', 'نفقة', 'عامل', 'عمل', 'راتب',
    'عقار', 'أرض', 'ملكية', 'بيع', 'شراء', 'إيجار', 'تأجير',
    'تعويض', 'ضرر', 'مسؤولية', 'أهلية', 'تقادم', 'تقاضي'
  ];
  
  // Check if question contains legal keywords
  for (const keyword of legalKeywords) {
    if (questionLower.includes(keyword)) {
      return true;
    }
  }
  
  // Check for question patterns
  const questionPatterns = [
    /ما (هو|هي|هذا|هذه|هؤلاء)/,
    /كم (هو|هي|هذا)/,
    /متى (يكون|يكون|هو)/,
    /كيف (يكون|هو|هي)/,
    /لماذا (يكون|هو|هي)/,
    /هل (يجوز|يجب|يحق|يحرم|يسمح|يمنع)/
  ];
  
  for (const pattern of questionPatterns) {
    if (pattern.test(questionLower)) {
      // If it's a question pattern, check if it's not just a greeting
      const greetingWords = ['ازيك', 'كيفك', 'كيف حالك', 'شكرا', 'شكراً', 'مرحبا', 'مرحباً'];
      const isGreeting = greetingWords.some(word => questionLower.includes(word));
      if (!isGreeting) {
        return true;
      }
    }
  }
  
  return false;
};

// Validate that question is appropriate
const validateQuestion = (question) => {
  const questionLower = question.toLowerCase();
  const arabicQuestion = question;
  
  // Keywords that indicate general/international law (should be rejected)
  const generalKeywords = [
    'قانون دولي', 'قانون عام', 'قوانين العالم', 'القانون الأمريكي',
    'القانون الأوروبي', 'القانون البريطاني', 'قانون أي دولة أخرى'
  ];
  
  // Check for general/international keywords first
  for (const keyword of generalKeywords) {
    if (arabicQuestion.includes(keyword)) {
      return {
        valid: false,
        error: 'عفواً، يجب أن يتعلق السؤال بالقانون المصري فقط. لا يمكننا الرد على استفسارات حول قوانين دول أخرى أو القانون الدولي.'
      };
    }
  }
  
  // Allow greetings, simple questions, and normal conversation
  // Only reject if it's clearly about non-Egyptian law
  return { valid: true };
};

export const sendMessage = async (req, res, next) => {
  try {
    const { message, chatId, selectedLaw, chatMode } = req.body;
    const userId = req.user._id;

    if (!message || !message.trim()) {
      return res.status(400).json({
        error: 'الرسالة مطلوبة.',
      });
    }

    // Validate question
    const validation = validateQuestion(message);
    if (!validation.valid) {
      return res.status(400).json({
        error: validation.error,
      });
    }

    let chat;
    
    // Get or create chat
    // Check if chatId is a valid MongoDB ObjectId (not a temp ID)
    const isValidObjectId = chatId && /^[0-9a-fA-F]{24}$/.test(chatId);
    
    if (chatId && isValidObjectId) {
      chat = await Chat.findOne({ _id: chatId, user: userId });
      if (!chat) {
        return res.status(404).json({
          error: 'المحادثة غير موجودة.',
        });
      }
    } else {
      // Create new chat (either no chatId provided or it's a temp ID)
      chat = new Chat({
        user: userId,
        title: message.substring(0, 50),
        preview: message.substring(0, 100),
        messages: [],
        selectedLaw: selectedLaw || null,
        chatMode: chatMode || 'general',
      });
    }

    // Update selectedLaw if provided in request
    if (selectedLaw !== undefined) {
      if (selectedLaw === null || selectedLaw === 'auto' || selectedLaw === '') {
        chat.selectedLaw = null; // Auto-detect
      } else {
        chat.selectedLaw = selectedLaw;
      }
    }

    // Update chatMode if provided in request
    if (chatMode !== undefined && chatMode) {
      chat.chatMode = chatMode;
    }

    // Check if this is a new chat with legal-advisor or business-advisor mode
    // Check before adding user message - use the updated chatMode
    const currentChatMode = chat.chatMode || chatMode || 'general';
    const isNewLegalAdvisorChat = chat.messages.length === 0 && currentChatMode === 'legal-advisor';
    const isNewBusinessAdvisorChat = chat.messages.length === 0 && currentChatMode === 'business-advisor';
    
    // If this is a new legal-advisor or business-advisor chat, send welcome message first
    // But also process the user message to get AI response
    if (isNewLegalAdvisorChat || isNewBusinessAdvisorChat) {
      let welcomeText = '';
      if (isNewLegalAdvisorChat) {
        welcomeText = 'أهلاً بك! أنا قانونك، مستشارك القانوني المتخصص في القانون المصري.\n\nإيه الموضوع اللي عايز استشارة فيه؟\n\nسأساعدك في فهم القضية بشكل كامل من خلال بعض الأسئلة، وبعدين هقدم لك تحليل قانوني شامل وحلول عملية.';
      } else {
        welcomeText = 'أهلاً بك! أنا قانونك، مستشارك القانوني المتخصص في القانون التجاري والأعمال.\n\nإيه الموضوع اللي عايز استشارة فيه؟ (تأسيس شركة، عقود، ضرائب، ملكية فكرية...)\n\nسأساعدك في فهم الموضوع بشكل كامل من خلال بعض الأسئلة، وبعدين هقدم لك تحليل قانوني شامل وحلول عملية.';
      }
      
      const welcomeMessage = {
        text: welcomeText,
        sender: 'bot',
        timestamp: new Date(),
      };
      
      // Add welcome message
      chat.messages.push(welcomeMessage);
      
      // Add user message as well (so it's saved in the chat)
      const userMessage = {
        text: message.trim(),
        sender: 'user',
        timestamp: new Date(),
      };
      chat.messages.push(userMessage);
      
      // Set title from first message
      chat.title = message.substring(0, 50);
      chat.preview = message.substring(0, 100);
      
      // Save chat with both welcome and user messages
      await chat.save();
      
      // Now process the user message to get AI response
      // Continue with the normal flow to get AI response
      // (Don't return here - let it continue to process the user message)
      // Note: user message is already added above, so skip adding it again below
    } else {
      // Add user message (for non-welcome message cases)
      chat.messages.push({
        text: message.trim(),
        sender: 'user',
        timestamp: new Date(),
      });
    }

    // Generate title from first message if it's a new chat
    const userMessagesCount = chat.messages.filter(m => m.sender === 'user').length;
    if (userMessagesCount === 1) {
      chat.title = message.substring(0, 50);
      chat.preview = message.substring(0, 100);
    }

    // Get AI response from Groq
    try {
      const groq = getGroqClient();
      
      // Load and search reference data for relevant information
      const referenceData = loadCachedReferenceData();
      
      // Determine which law to search in
      let lawToSearch = null;
      
      if (chat.selectedLaw && chat.selectedLaw !== 'auto') {
        // Use explicitly selected law
        lawToSearch = chat.selectedLaw;
        console.log(`🎯 Using explicitly selected law: ${lawToSearch}`);
      } else {
        // Auto-detect from context
        const conversationHistory = chat.messages
          .slice(0, -1) // Exclude current message
          .map(msg => msg.text)
          .join(' ');
        
        const detectedLaws = autoDetectRelevantLaws(message, referenceData, [conversationHistory]);
        
        if (detectedLaws.length > 0) {
          lawToSearch = detectedLaws[0]; // Use most relevant
          console.log(`🔍 Auto-detected law: ${lawToSearch} (from: ${detectedLaws.join(', ')})`);
          
          // Update chat with detected law for future messages
          if (!chat.selectedLaw) {
            chat.selectedLaw = lawToSearch;
          }
        } else {
          console.log('🔍 No specific law detected, searching all laws');
        }
      }
      
      // Search with law filter
      const relevantReferences = searchReferenceData(message, referenceData, 5, lawToSearch);
      const { context: referenceContext, sources } = formatReferenceForPrompt(relevantReferences);
      
      // Determine if this is a legal question
      const isLegalQuestion = isQuestionLegal(message);
      
      // Search external sources if needed (for legal questions)
      let externalSources = [];
      let externalContext = '';
      let externalSourcesList = [];
      
      if (isLegalQuestion && shouldUseExternalSearch(message, relevantReferences.length > 0)) {
        try {
          externalSources = await searchExternalSources(message, 3);
          if (externalSources.length > 0) {
            const externalFormatted = formatExternalSourcesForPrompt(externalSources);
            externalContext = externalFormatted.context;
            externalSourcesList = externalFormatted.sources;
          }
        } catch (error) {
          console.error('Error searching external sources:', error);
          // Continue without external sources if search fails
        }
      }
      
      // Build system prompt based on chat mode (use already determined mode)
      const shouldIncludeReferences = isLegalQuestion && (referenceContext || externalContext);
      
      let systemPrompt = getSystemPromptForMode(
        currentChatMode,
        shouldIncludeReferences ? referenceContext : '',
        shouldIncludeReferences ? externalContext : ''
      );
      
      // For legal-advisor and business-advisor modes, add special instructions for interactive questioning
      if (currentChatMode === 'legal-advisor' || currentChatMode === 'business-advisor') {
        const userMessagesCount = chat.messages.filter(m => m.sender === 'user').length;
        
        // Check if we have enough information (more than 3 user messages)
        const hasEnoughInfo = userMessagesCount > 3;
        
        const modeName = currentChatMode === 'legal-advisor' ? 'المستشار القانوني' : 'مستشار الأعمال';
        
        if (userMessagesCount <= 1) {
          // This is the first user message
          systemPrompt += `\n\n**⚠️ ملاحظة حرجة جداً:** هذه بداية المحادثة في وضع ${modeName}. يجب أن تبدأ مباشرة بأسئلة استكشافية واحدة تلو الأخرى. لا تعطي إجابة مباشرة أو تحليل كامل الآن. اسأل سؤال واحد فقط متعلق بالموضوع الذي ذكره المستخدم وانتظر إجابته.`;
        } else if (!hasEnoughInfo) {
          // Continue asking questions - we need more information
          systemPrompt += '\n\n**⚠️ ملاحظة مهمة:** استمر في طرح أسئلة استكشافية واحدة تلو الأخرى بناءً على الإجابات السابقة. اسأل سؤال واحد فقط في كل رسالة. لا تعطي تحليل كامل الآن - نحتاج لمزيد من المعلومات. استخدم المعلومات التي جمعتها لتوجيه السؤال التالي.';
        } else {
          // We have enough information - provide analysis
          const analysisType = currentChatMode === 'legal-advisor' 
            ? 'تحليل القضية، الخطوات العملية المطلوبة، المستندات المطلوبة، التوقيتات المهمة، والمخاطر المحتملة'
            : 'تحليل القضية، الخطوات العملية المطلوبة، المستندات المطلوبة، القوانين واللوائح ذات الصلة، والنصائح الوقائية';
          systemPrompt += `\n\n**✅ ملاحظة:** لقد جمعت معلومات كافية عن الموضوع. الآن قدم تحليل قانوني شامل يتضمن: ${analysisType}.`;
        }
      }
      
      // Build conversation history
      const conversationMessages = [
        {
          role: 'system',
          content: systemPrompt
        }
      ];

      // Add previous messages (excluding the current one we just added)
      chat.messages.slice(0, -1).forEach(msg => {
        if (msg.sender === 'user') {
          conversationMessages.push({ role: 'user', content: msg.text });
        } else if (msg.sender === 'bot' && !msg.isError) {
          conversationMessages.push({ role: 'assistant', content: msg.text });
        }
      });

      // Add current user message
      conversationMessages.push({ role: 'user', content: message.trim() });

      // Define model fallback list (in order of preference)
      const modelFallbackList = [
        process.env.GROQ_MODEL, // User's preferred model from .env
        'meta-llama/llama-4-maverick-17b-128e-instruct',
        'meta-llama/llama-4-scout-17b-16e-instruct',
        'qwen/qwen3-32b',
        'openai/gpt-oss-120b',
        'openai/gpt-oss-20b',
        'llama-3.1-8b-instant',
        'meta-llama/llama-guard-4-12b',
        'openai/gpt-oss-safeguard-20b',
        'llama-3.3-70b-versatile', // Last resort fallback
      ].filter(Boolean); // Remove undefined/null values

      // Try models in order until one works
      let aiResponse = null;
      let lastError = null;
      
      for (const model of modelFallbackList) {
        try {
          console.log(`🔄 Attempting to use model: ${model}`);
          const completion = await groq.chat.completions.create({
            messages: conversationMessages,
            model: model,
            temperature: 0.7,
            max_tokens: 2000,
          });
          
          aiResponse = completion.choices[0]?.message?.content;
          if (aiResponse) {
            console.log(`✅ Successfully used model: ${model}`);
            break;
          }
        } catch (error) {
          console.warn(`⚠️  Model ${model} failed:`, error.message);
          lastError = error;
          // Continue to next model
          continue;
        }
      }

      // If all models failed, use error message
      if (!aiResponse) {
        console.error('❌ All models failed. Last error:', lastError?.message);
        aiResponse = 'عذراً، لم أتمكن من معالجة سؤالك. يرجى المحاولة مرة أخرى.';
      }

      // If it's a legal question and we have sources, ensure sources are included
      if (isLegalQuestion) {
        let sourcesText = '';
        
        // Add local sources
        if (sources && sources.length > 0) {
          sourcesText += formatSourcesForDisplay(sources);
        }
        
        // Add external sources
        if (externalSourcesList && externalSourcesList.length > 0) {
          sourcesText += formatExternalSourcesForDisplay(externalSourcesList);
        }
        
        // Check if sources are already in the response
        const hasSources = aiResponse.includes('المصادر:') || 
                          aiResponse.includes('المصدر:') ||
                          aiResponse.toLowerCase().includes('مصدر');
        
        if (!hasSources && sourcesText) {
          // Add sources at the end
          aiResponse += sourcesText;
        }
      }

      // Add bot response
      chat.messages.push({
        text: aiResponse,
        sender: 'bot',
        timestamp: new Date(),
        sources: isLegalQuestion && sources && sources.length > 0 ? sources : null,
        externalSources: isLegalQuestion && externalSourcesList && externalSourcesList.length > 0 ? externalSourcesList : null,
      });
    } catch (groqError) {
      console.error('Groq API Error:', groqError);
      chat.messages.push({
        text: 'عذراً، حدث خطأ أثناء معالجة سؤالك. يرجى المحاولة مرة أخرى.',
        sender: 'bot',
        timestamp: new Date(),
        isError: true,
      });
    }

    // Save chat
    await chat.save();

    // Update user's chats array (keep only 10 most recent)
    const user = await User.findById(userId);
    if (!user.chats.includes(chat._id)) {
      user.chats.unshift(chat._id);
      // Keep only 10 chats
      if (user.chats.length > 10) {
        // Remove oldest chats
        const chatsToRemove = user.chats.slice(10);
        await Chat.deleteMany({ _id: { $in: chatsToRemove } });
        user.chats = user.chats.slice(0, 10);
      }
      await user.save();
    }

    // Return the last bot message (AI response)
    const lastBotMessage = chat.messages.filter(m => m.sender === 'bot').pop();
    
    res.json({
      response: lastBotMessage ? lastBotMessage.text : chat.messages[chat.messages.length - 1].text,
      chatId: chat._id,
      message: lastBotMessage || chat.messages[chat.messages.length - 1],
    });
  } catch (error) {
    next(error);
  }
};

export const getChats = async (req, res, next) => {
  try {
    const userId = req.user._id;

    const chats = await Chat.find({ user: userId })
      .sort({ createdAt: -1 })
      .limit(10)
      .select('title preview messages createdAt updatedAt');

    res.json({
      chats: chats.map(chat => ({
        id: chat._id.toString(),
        title: chat.title,
        preview: chat.preview,
        messages: chat.messages,
        selectedLaw: chat.selectedLaw || null,
        chatMode: chat.chatMode || 'general',
        createdAt: chat.createdAt,
        updatedAt: chat.updatedAt,
      })),
    });
  } catch (error) {
    next(error);
  }
};

export const getChat = async (req, res, next) => {
  try {
    const { chatId } = req.params;
    const userId = req.user._id;

    const chat = await Chat.findOne({ _id: chatId, user: userId });

    if (!chat) {
      return res.status(404).json({
        error: 'المحادثة غير موجودة.',
      });
    }

    res.json({
      chat: {
        id: chat._id.toString(),
        title: chat.title,
        preview: chat.preview,
        messages: chat.messages,
        selectedLaw: chat.selectedLaw || null,
        chatMode: chat.chatMode || 'general',
        createdAt: chat.createdAt,
        updatedAt: chat.updatedAt,
      },
    });
  } catch (error) {
    next(error);
  }
};

export const deleteChat = async (req, res, next) => {
  try {
    const { chatId } = req.params;
    const userId = req.user._id;

    const chat = await Chat.findOne({ _id: chatId, user: userId });

    if (!chat) {
      return res.status(404).json({
        error: 'المحادثة غير موجودة.',
      });
    }

    await Chat.deleteOne({ _id: chatId });

    // Remove from user's chats array
    await User.findByIdAndUpdate(userId, {
      $pull: { chats: chatId },
    });

    res.json({
      message: 'تم حذف المحادثة بنجاح.',
    });
  } catch (error) {
    next(error);
  }
};

/**
 * Get list of available laws
 */
export const getAvailableLawsList = async (req, res, next) => {
  try {
    const laws = getAvailableLaws();
    res.json({
      laws: laws,
      count: laws.length,
    });
  } catch (error) {
    next(error);
  }
};

/**
 * Update chat's mode
 */
export const updateChatMode = async (req, res, next) => {
  try {
    const { chatId } = req.params;
    const { chatMode } = req.body;
    const userId = req.user._id;

    const chat = await Chat.findOne({ _id: chatId, user: userId });

    if (!chat) {
      return res.status(404).json({
        error: 'المحادثة غير موجودة.',
      });
    }

    // Validate chatMode
    const validModes = ['general', 'legal-advisor', 'business-advisor', 'legal-explainer', 
                        'contract-drafting', 'risk-assessment', 'family-law', 'criminal-law'];
    
    if (chatMode && !validModes.includes(chatMode)) {
      return res.status(400).json({
        error: 'وضع المحادثة غير صحيح.',
      });
    }

    chat.chatMode = chatMode || 'general';
    await chat.save();

    res.json({
      message: 'تم تحديث وضع المحادثة بنجاح.',
      chat: {
        id: chat._id.toString(),
        chatMode: chat.chatMode,
      },
    });
  } catch (error) {
    next(error);
  }
};

/**
 * Update chat's selected law
 */
export const updateChatLaw = async (req, res, next) => {
  try {
    const { chatId } = req.params;
    const { selectedLaw } = req.body;
    const userId = req.user._id;

    const chat = await Chat.findOne({ _id: chatId, user: userId });

    if (!chat) {
      return res.status(404).json({
        error: 'المحادثة غير موجودة.',
      });
    }

    // Validate selectedLaw (can be null for auto-detect, or a valid law name)
    if (selectedLaw !== null && selectedLaw !== 'auto' && selectedLaw !== '') {
      const referenceData = loadCachedReferenceData();
      const isValidLaw = referenceData.some(law => 
        law.lawName === selectedLaw || law.filename === selectedLaw
      );
      
      if (!isValidLaw) {
        return res.status(400).json({
          error: 'القانون المحدد غير موجود.',
        });
      }
    }

    chat.selectedLaw = selectedLaw === 'auto' || selectedLaw === '' ? null : selectedLaw;
    await chat.save();

    res.json({
      message: 'تم تحديث القانون المحدد بنجاح.',
      chat: {
        id: chat._id.toString(),
        selectedLaw: chat.selectedLaw,
      },
    });
  } catch (error) {
    next(error);
  }
};

