import axios from 'axios';

/**
 * Search external sources for legal information
 * Uses Google Custom Search API for reliable sources
 */
export const searchExternalSources = async (query, maxResults = 3) => {
  const apiKey = process.env.GOOGLE_SEARCH_API_KEY;
  const searchEngineId = process.env.GOOGLE_SEARCH_ENGINE_ID;

  // If API keys are not configured, return empty results
  if (!apiKey || !searchEngineId) {
    console.warn('⚠️  Google Search API keys not configured. External search disabled.');
    return [];
  }

  try {
    // Build search query - focus on Egyptian law sources
    // Add site restrictions to trusted legal sources
    const searchQuery = `${query} قانون مصري site:moj.gov.eg OR site:manshurat.gov.eg OR site:alexu.edu.eg OR "قانون مصري"`;
    
    const response = await axios.get('https://www.googleapis.com/customsearch/v1', {
      params: {
        key: apiKey,
        cx: searchEngineId,
        q: searchQuery,
        num: maxResults,
        lr: 'lang_ar', // Arabic language
        safe: 'active',
      },
      timeout: 5000, // 5 second timeout
    });

    if (!response.data || !response.data.items) {
      return [];
    }

    const results = response.data.items.map((item, index) => ({
      title: item.title,
      url: item.link,
      snippet: item.snippet,
      displayUrl: item.displayLink,
      rank: index + 1,
    }));

    console.log(`✅ Found ${results.length} external search results for: ${query.substring(0, 50)}`);
    return results;
  } catch (error) {
    // Log error but don't fail the request
    console.error('❌ External search error:', error.message);
    if (error.response) {
      console.error('Response status:', error.response.status);
      console.error('Response data:', error.response.data);
    }
    return [];
  }
};

/**
 * Format external search results for inclusion in AI prompt
 */
export const formatExternalSourcesForPrompt = (externalResults) => {
  if (!externalResults || externalResults.length === 0) {
    return { context: '', sources: [] };
  }

  let formattedText = '\n\n=== مصادر خارجية موثوقة ===\n\n';
  const sources = [];

  externalResults.forEach((item, index) => {
    formattedText += `[مصدر خارجي ${index + 1}]\n`;
    formattedText += `العنوان: ${item.title}\n`;
    formattedText += `الرابط: ${item.url}\n`;
    formattedText += `الملخص: ${item.snippet}\n\n`;

    sources.push({
      title: item.title,
      url: item.url,
      displayUrl: item.displayUrl,
      snippet: item.snippet,
    });
  });

  formattedText += '=== نهاية المصادر الخارجية ===\n\n';
  formattedText += 'ملاحظات مهمة:\n';
  formattedText += '- يمكنك استخدام هذه المصادر الخارجية كمراجع إضافية موثوقة\n';
  formattedText += '- تأكد من أن المعلومات من المصادر الخارجية تتوافق مع القانون المصري\n';
  formattedText += '- في نهاية إجابتك، يمكنك ذكر المصادر الخارجية المستخدمة\n';

  return { context: formattedText, sources };
};

/**
 * Format external sources for display at the end of answer
 */
export const formatExternalSourcesForDisplay = (externalSources) => {
  if (!externalSources || externalSources.length === 0) {
    return '';
  }

  let sourcesText = '\n\n🌐 **مصادر خارجية موثوقة:**\n';

  externalSources.forEach((source, index) => {
    sourcesText += `${index + 1}. [${source.title}](${source.url})\n`;
  });

  return sourcesText;
};

/**
 * Check if external search should be used for this query
 * Only use for legal questions that might need additional context
 */
export const shouldUseExternalSearch = (query, hasLocalReferences) => {
  // Use external search if:
  // 1. It's a legal question
  // 2. Either we don't have local references OR the query seems complex
  const isComplexQuery = query.length > 50 || 
                        query.includes('؟') || 
                        query.includes('ماذا') ||
                        query.includes('كيف') ||
                        query.includes('متى');

  return isComplexQuery || !hasLocalReferences;
};

