import fs from 'fs';
import path from 'path';
import { fileURLToPath } from 'url';
import { dirname } from 'path';

const __filename = fileURLToPath(import.meta.url);
const __dirname = dirname(__filename);

// Path to data directory
const DATA_DIR = path.join(__dirname, '..', 'data');

/**
 * Load all reference data files from the data directory
 * Supports: .txt, .md, .json files
 */
export const loadReferenceData = () => {
  const referenceData = [];
  
  try {
    // Check if data directory exists
    if (!fs.existsSync(DATA_DIR)) {
      console.warn(`⚠️  Data directory not found: ${DATA_DIR}`);
      return referenceData;
    }

    // Read all files in data directory
    const files = fs.readdirSync(DATA_DIR);
    
    for (const file of files) {
      const filePath = path.join(DATA_DIR, file);
      const stat = fs.statSync(filePath);
      
      // Skip directories and hidden files
      if (stat.isDirectory() || file.startsWith('.')) {
        continue;
      }

      // Only process text-based files
      const ext = path.extname(file).toLowerCase();
      if (!['.txt', '.md', '.json'].includes(ext)) {
        continue;
      }

      try {
        const content = fs.readFileSync(filePath, 'utf-8');
        
        // Parse JSON files, keep text/markdown as-is
        let parsedContent = content;
        if (ext === '.json') {
          try {
            parsedContent = JSON.parse(content);
            // If it's an array or object, stringify it for easier searching
            if (typeof parsedContent === 'object') {
              parsedContent = JSON.stringify(parsedContent, null, 2);
            }
          } catch (e) {
            console.warn(`⚠️  Failed to parse JSON file ${file}:`, e.message);
            continue;
          }
        }

        // Extract law name from filename (remove extension)
        const lawName = file.replace(/\.[^/.]+$/, '');

        referenceData.push({
          filename: file,
          lawName: lawName,
          content: parsedContent,
          size: content.length,
        });
      } catch (error) {
        console.error(`❌ Error reading file ${file}:`, error.message);
      }
    }

    console.log(`✅ Loaded ${referenceData.length} reference law file(s)`);
    referenceData.forEach(item => {
      console.log(`   - ${item.lawName}`);
    });
    return referenceData;
  } catch (error) {
    console.error('❌ Error loading reference data:', error.message);
    return referenceData;
  }
};

/**
 * Get list of available laws
 * Returns array of law names and filenames
 */
export const getAvailableLaws = () => {
  const referenceData = loadReferenceData();
  return referenceData.map(law => ({
    name: law.lawName,
    filename: law.filename,
  }));
};

/**
 * Auto-detect which law(s) are most relevant based on query and conversation context
 * Returns array of law names (can be multiple)
 */
export const autoDetectRelevantLaws = (query, referenceData, conversationHistory = []) => {
  if (!referenceData || referenceData.length === 0) {
    return [];
  }

  const queryLower = query.toLowerCase();
  const allText = (conversationHistory.join(' ') + ' ' + query).toLowerCase();
  
  // Law-specific keywords mapping
  const lawKeywords = {
    'عقوبات': ['عقوبات', 'جريمة', 'عقوبة', 'سجن', 'غرامة', 'حكم', 'محكمة جزائية'],
    'مدني': ['مدني', 'عقد', 'بيع', 'شراء', 'إيجار', 'ملكية', 'تعويض', 'ضرر'],
    'تجاري': ['تجاري', 'تجارة', 'شركة', 'سجل تجاري', 'فاتورة', 'شيك', 'كمبيالة'],
    'أحوال شخصية': ['أحوال شخصية', 'زواج', 'طلاق', 'نفقة', 'حضانة', 'ميراث'],
    'ميراث': ['ميراث', 'ورثة', 'وصية', 'تركة', 'إرث'],
    'دستور': ['دستور', 'دستوري', 'حقوق', 'حريات', 'مواطنة'],
    'إرهاب': ['إرهاب', 'إرهابي', 'مكافحة الإرهاب'],
    'مجلس الدولة': ['مجلس الدولة', 'قضاء إداري', 'قرار إداري'],
  };

  const lawScores = {};
  
  // Initialize scores
  referenceData.forEach(law => {
    lawScores[law.lawName] = 0;
  });

  // Score based on keywords
  Object.keys(lawKeywords).forEach(keyword => {
    const laws = referenceData.filter(law => 
      law.lawName.toLowerCase().includes(keyword.toLowerCase())
    );
    
    lawKeywords[keyword].forEach(kw => {
      const matches = (allText.match(new RegExp(kw, 'gi')) || []).length;
      laws.forEach(law => {
        lawScores[law.lawName] += matches * 10;
      });
    });
  });

  // Score based on direct mentions of law names
  referenceData.forEach(law => {
    const lawNameLower = law.lawName.toLowerCase();
    const matches = (allText.match(new RegExp(lawNameLower.replace(/[.*+?^${}()|[\]\\]/g, '\\$&'), 'gi')) || []).length;
    lawScores[law.lawName] += matches * 20;
  });

  // Get top scoring laws (threshold: at least 10 points)
  const relevantLaws = Object.entries(lawScores)
    .filter(([_, score]) => score >= 10)
    .sort(([_, a], [__, b]) => b - a)
    .slice(0, 2) // Top 2 most relevant
    .map(([lawName, _]) => lawName);

  return relevantLaws;
};

/**
 * Extract article numbers and content from law text
 * Looks for patterns like "المادة (1)", "المادة 1", etc.
 */
const extractArticles = (content) => {
  const articles = [];
  // Match patterns like: المادة (1), المادة 1, المادة الأولى, etc.
  const articlePattern = /المادة\s*(?:\(|[\d\u0660-\u0669]+|الأولى|الثانية|الثالثة|الرابعة|الخامسة|السادسة|السابعة|الثامنة|التاسعة|العاشرة)/gi;
  
  const lines = content.split('\n');
  let currentArticle = null;
  let articleText = '';

  for (let i = 0; i < lines.length; i++) {
    const line = lines[i].trim();
    
    if (articlePattern.test(line)) {
      // Save previous article if exists
      if (currentArticle && articleText) {
        articles.push({
          number: currentArticle,
          text: articleText.trim(),
        });
      }
      
      // Start new article
      currentArticle = line;
      articleText = line + '\n';
    } else if (currentArticle && line) {
      articleText += line + '\n';
    }
  }
  
  // Add last article
  if (currentArticle && articleText) {
    articles.push({
      number: currentArticle,
      text: articleText.trim(),
    });
  }
  
  return articles;
};

/**
 * Search for relevant content in reference data based on keywords
 * Returns the most relevant chunks with source information
 * @param {string} query - The search query
 * @param {Array} referenceData - All reference data files
 * @param {number} maxResults - Maximum number of results to return
 * @param {string|null} filterByLaw - Optional: filter by specific law name (filename or lawName)
 */
export const searchReferenceData = (query, referenceData, maxResults = 5, filterByLaw = null) => {
  if (!referenceData || referenceData.length === 0) {
    return [];
  }

  const queryLower = query.toLowerCase();
  const queryWords = queryLower.split(/\s+/).filter(word => word.length > 2);
  
  // If query is too short, return empty
  if (queryWords.length === 0) {
    return [];
  }
  
  const results = [];
  
  // Filter reference data by law if specified
  let dataToSearch = referenceData;
  if (filterByLaw) {
    const filterLower = filterByLaw.toLowerCase();
    dataToSearch = referenceData.filter(lawFile => 
      lawFile.lawName.toLowerCase().includes(filterLower) ||
      lawFile.filename.toLowerCase().includes(filterLower)
    );
    
    // If no matching law found, search all (fallback)
    if (dataToSearch.length === 0) {
      console.warn(`⚠️  Law filter "${filterByLaw}" not found, searching all laws`);
      dataToSearch = referenceData;
    } else {
      console.log(`🔍 Filtering search to: ${dataToSearch.map(l => l.lawName).join(', ')}`);
    }
  }
  
  // Search in each law file
  for (const lawFile of dataToSearch) {
    const contentLower = lawFile.content.toLowerCase();
    let score = 0;
    let matchedArticles = [];
    
    // Extract articles from this law
    const articles = extractArticles(lawFile.content);
    
    // Score based on full query match
    const phraseMatches = (contentLower.match(new RegExp(queryLower.replace(/[.*+?^${}()|[\]\\]/g, '\\$&'), 'gi')) || []).length;
    score += phraseMatches * 20;
    
    // Score based on individual word matches
    queryWords.forEach(word => {
      const wordMatches = (contentLower.match(new RegExp(`\\b${word}\\b`, 'gi')) || []).length;
      score += wordMatches * 3;
    });
    
    // Check each article for relevance
    articles.forEach(article => {
      const articleLower = article.text.toLowerCase();
      let articleScore = 0;
      
      // Check for phrase match in article
      if (articleLower.includes(queryLower)) {
        articleScore += 15;
      }
      
      // Check for word matches in article
      queryWords.forEach(word => {
        if (articleLower.includes(word)) {
          articleScore += 2;
        }
      });
      
      if (articleScore > 0) {
        matchedArticles.push({
          ...article,
          score: articleScore,
        });
      }
    });
    
    // Sort articles by score and take top 2 per law
    matchedArticles.sort((a, b) => b.score - a.score);
    matchedArticles = matchedArticles.slice(0, 2);
    
    // If we have matching articles, add them
    if (matchedArticles.length > 0) {
      matchedArticles.forEach(article => {
        results.push({
          lawName: lawFile.lawName,
          filename: lawFile.filename,
          article: article.number,
          content: article.text,
          score: article.score + (score / 10), // Boost score if law file is relevant
        });
      });
    } else if (score > 0) {
      // If no specific articles but law is relevant, add relevant snippet
      const snippet = extractRelevantSnippet(lawFile.content, queryWords, 300);
      if (snippet) {
        results.push({
          lawName: lawFile.lawName,
          filename: lawFile.filename,
          article: null,
          content: snippet,
          score: score,
        });
      }
    }
  }

  // Sort by score and return top results
  const topResults = results
    .sort((a, b) => b.score - a.score)
    .slice(0, maxResults);

  return topResults;
};

/**
 * Extract relevant snippet from content based on query words
 */
const extractRelevantSnippet = (content, queryWords, maxLength = 300) => {
  const lines = content.split('\n');
  let bestLine = '';
  let bestScore = 0;
  
  for (const line of lines) {
    if (line.trim().length < 20) continue; // Skip very short lines
    
    const lineLower = line.toLowerCase();
    let score = 0;
    
    queryWords.forEach(word => {
      if (lineLower.includes(word)) {
        score += 1;
      }
    });
    
    if (score > bestScore) {
      bestScore = score;
      bestLine = line;
    }
  }
  
  if (bestLine) {
    // Try to get context around the best line
    const bestIndex = lines.indexOf(bestLine);
    const start = Math.max(0, bestIndex - 1);
    const end = Math.min(lines.length, bestIndex + 3);
    const snippet = lines.slice(start, end).join('\n');
    
    return snippet.length > maxLength 
      ? snippet.substring(0, maxLength) + '...'
      : snippet;
  }
  
  return null;
};

/**
 * Format reference data for inclusion in AI prompt
 * Returns formatted text and sources array
 */
export const formatReferenceForPrompt = (referenceResults) => {
  if (!referenceResults || referenceResults.length === 0) {
    return { context: '', sources: [] };
  }

  let formattedText = '\n\n=== المراجع القانونية المصرية الموثوقة ===\n\n';
  const sources = [];
  
  referenceResults.forEach((item, index) => {
    formattedText += `[مرجع ${index + 1}]\n`;
    
    if (item.article) {
      formattedText += `القانون: ${item.lawName}\n`;
      formattedText += `${item.article}\n`;
      formattedText += `${item.content}\n\n`;
      
      sources.push({
        law: item.lawName,
        article: item.article,
        filename: item.filename,
      });
    } else {
      formattedText += `القانون: ${item.lawName}\n`;
      formattedText += `${item.content}\n\n`;
      
      sources.push({
        law: item.lawName,
        article: null,
        filename: item.filename,
      });
    }
  });

  formattedText += '=== نهاية المراجع القانونية ===\n\n';
  formattedText += 'تعليمات مهمة:\n';
  formattedText += '- يجب أن تعتمد إجابتك بشكل كامل على المراجع القانونية المذكورة أعلاه\n';
  formattedText += '- في نهاية إجابتك، يجب أن تذكر المصادر المستخدمة بالتنسيق التالي:\n';
  formattedText += '  "المصادر:\n';
  formattedText += '  - [اسم القانون] - [رقم المادة إن وجدت]"\n';
  formattedText += '- إذا لم تجد المعلومات في المراجع، أخبر المستخدم بذلك بوضوح\n';
  formattedText += '- لا تخترع معلومات قانونية غير موجودة في المراجع\n';
  
  return { context: formattedText, sources };
};

/**
 * Format sources for display at the end of answer
 */
export const formatSourcesForDisplay = (sources) => {
  if (!sources || sources.length === 0) {
    return '';
  }
  
  let sourcesText = '\n\n📚 **المصادر:**\n';
  
  sources.forEach((source, index) => {
    sourcesText += `${index + 1}. ${source.law}`;
    if (source.article) {
      sourcesText += ` - ${source.article}`;
    }
    sourcesText += '\n';
  });
  
  return sourcesText;
};

